### Encoding: UTF-8

### Name: smooth.roc
### Title: Smooth a ROC curve
### Aliases: smooth smooth.roc smooth.smooth.roc smooth.default
### Keywords: univar nonparametric utilities roc smooth

### ** Examples

data(aSAH)

## Basic example

rocobj <- roc(aSAH$outcome, aSAH$s100b)
smooth(rocobj)
# or directly with roc()
roc(aSAH$outcome, aSAH$s100b, smooth=TRUE)

# plotting
plot(rocobj)
rs <- smooth(rocobj, method="binormal")
plot(rs, add=TRUE, col="green")
rs2 <- smooth(rocobj, method="density")
plot(rs2, add=TRUE, col="blue")
rs3 <- smooth(rocobj, method="fitdistr", density="lognormal")
plot(rs3, add=TRUE, col="magenta")
legend(.6, .4, legend=c("Empirical", "Binormal", "Density", "Log-normal"),
       col=c("black", "green", "blue", "magenta"), lwd=2)

## Advanced smoothing

# different distibutions for controls and cases:
smooth(rocobj, method="fitdistr", density.controls="normal", density.cases="lognormal")

# with densities
width <- bandwidth.nrd(rocobj$predictor)
density.controls <- density(rocobj$controls, from=min(rocobj$predictor) - 3 * width,
                            to=max(rocobj$predictor) + 3*width, width=width, window="gaussian")
density.cases <- density(rocobj$cases, from=min(rocobj$predictor) - 3 * width,
                            to=max(rocobj$predictor) + 3*width, width=width, window="gaussian")
smooth(rocobj, method="density", density.controls=density.controls$y, 
       density.cases=density.cases$y)
# which is roughly what is done by a simple:
smooth(rocobj, method="density")

## Smoothing artificial ROC curves

# two normals
roc.norm <- roc(rep(c(0, 1), each=1000), 
                c(rnorm(1000), rnorm(1000)+1), plot=TRUE)
plot(smooth(roc.norm), col="green", lwd=1, add=TRUE)
plot(smooth(roc.norm, method="density"), col="red", lwd=1, add=TRUE)
plot(smooth(roc.norm, method="fitdistr"), col="blue", lwd=1, add=TRUE)
legend(.6, .4, legend=c("empirical", "binormal", "density",, "fitdistr"),
       col=c(par("fg"), "green", "red", "blue"), lwd=c(2, 1, 1, 1, 1))
       
# deviation from the normality
roc.norm.exp <- roc(rep(c(0, 1), each=1000), 
                    c(rnorm(1000), rexp(1000)), plot=TRUE)
plot(smooth(roc.norm.exp), col="green", lwd=1, add=TRUE)
plot(smooth(roc.norm.exp, method="density"), col="red", lwd=1, add=TRUE)
# Wrong fitdistr: normality assumed by default
plot(smooth(roc.norm.exp, method="fitdistr"), col="blue", lwd=1, add=TRUE)
# Correct fitdistr
plot(smooth(roc.norm.exp, method="fitdistr", density.controls="normal",
            density.cases="exponential"), col="purple", lwd=1, add=TRUE)
legend(.6, .4, legend=c("empirical", "binormal", "density",
                        "wrong fitdistr", "correct fitdistr"),
       col=c(par("fg"), "green", "red", "blue", "purple"), lwd=c(2, 1, 1, 1, 1))

# large deviation from the normality
roc.unif.exp <- roc(rep(c(0, 1), each=1000), 
                    c(runif(1000, -1, 1), rexp(1000)), plot=TRUE)
plot(smooth(roc.unif.exp), col="green", lwd=1, add=TRUE)
plot(smooth(roc.unif.exp, method="density"), col="red", lwd=1, add=TRUE)
# Wrong fitdistr: normality assumed by default
plot(smooth(roc.unif.exp, method="fitdistr"), col="blue", lwd=1, add=TRUE)
# Correct fitdistr
plot(smooth(roc.unif.exp, method="fitdistr", density.controls="uniform",
            density.cases="exponential"), col="purple", lwd=1, add=TRUE)
legend(.6, .4, legend=c("empirical", "binormal", "density", "density ucv",
                        "wrong fitdistr", "correct fitdistr"),
       col=c(par("fg"), "green", "red", "magenta", "blue", "purple"),
lwd=c(2, 1, 1, 1, 1))

# 2 uniform distributions with a custom density function
unif.density <- function(x, n, from, to, bw, kernel, ...) {
  smooth.x <- seq(from=from, to=to, length.out=n)
  smooth.y <- dunif(smooth.x, min=min(x), max=max(x))
  return(smooth.y)
}
roc.unif <- roc(rep(c(0, 1), each=1000),
                c(runif(1000, -1, 1), runif(1000, 0, 2)), plot=TRUE)
s <- smooth(roc.unif, method="density", density=unif.density)
plot(roc.unif)
plot(s, add=TRUE, col="grey")

## Not run: 
##D # you can bootstrap a ROC curve smoothed with a density function:
##D ci(s, boot.n=100)
## End(Not run)



