\encoding{UTF-8}
\name{ci.thresholds}
\alias{ci.thresholds}
\alias{ci.thresholds.default}
\alias{ci.thresholds.formula}
\alias{ci.thresholds.roc}
\alias{ci.thresholds.smooth.roc}

\title{
  Compute the confidence interval of thresholds
}

\description{
  This function computes the confidence interval (CI) of the sensitivity
  and specificity of the thresholds given in argument.
  By default, the 95\% CI are computed with 2000 stratified bootstrap replicates.
}

\usage{
ci.thresholds(x, ...)
\S3method{ci.thresholds}{roc}(roc, conf.level=0.95, boot.n=2000,
boot.stratified=TRUE, thresholds = "local maximas", ...) 
\S3method{ci.thresholds}{smooth.roc}(smooth.roc, ...)
\S3method{ci.thresholds}{formula}(formula, data, ...)
\S3method{ci.thresholds}{default}(response, predictor, ...)
}
		   
\arguments{
  \item{x}{a roc object from the \link{roc} function (for ci.thresholds.roc),
    a formula (for ci.thresholds.formula) or a response vector (for
    ci.thresholds.default).
  }
  \item{roc}{a \dQuote{roc} object from the \code{\link{roc}} function.}
  \item{smooth.roc}{not available for \link[=smooth.roc]{smoothed} ROC
    curves, available only to catch the error and provide a clear error
    message.
  }
  \item{response, predictor}{arguments for the \code{\link{roc}} function.}
  \item{formula, data}{a formula (and possibly a data object) of type
	response~predictor for the \code{\link{roc}} function.
  }
  \item{conf.level}{the width of the confidence interval as [0,1], never
  in percent. Default: 0.95, resulting in a 95\% CI.
  }
  \item{boot.n}{the number of bootstrap replicates. Default: 2000.}
  \item{boot.stratified}{should the bootstrap be stratified (default, same number
	of cases/controls in each replicate than in the original sample) or
	not.
  }
  \item{thresholds}{on which thresholds to evaluate the CI. Either the
	numeric values of the thresholds, a logical vector (as index of
	\code{roc$thresholds}) or a character \dQuote{all}, \dQuote{local
	  maximas} or \dQuote{best}.
  }
  \item{\dots}{further arguments passed to or from other methods,
    especially arguments for \code{\link{roc}} and \code{ci.thresholds.roc}
    when calling \code{ci.thresholds.default} or \code{ci.thresholds.formula}.
  }
}

\details{
  \code{ci.thresholds.formula} and \code{ci.thresholds.default} are convenience methods
  that build the ROC curve (with the \code{\link{roc}} function) before
  calling \code{ci.thresholds.roc}. You can pass them arguments for both
  \code{\link{roc}} and \code{ci.thresholds.roc}. Simply use \code{ci.thresholds}
  that will dispatch to the correct method.

  This function creates \code{boot.n} bootstrap replicate of the ROC
  curve, and evaluates the sensitivity and specificity at thresholds
  given by the \code{thresholds} argument. Then it computes the
  confidence interval as the percentiles given by \code{conf.level}.

  Stratification of bootstrap can be controlled
  with \code{boot.stratified}. In stratified bootstrap, each replicate
  contains the same number of cases and controls than the original
  sample. Stratification is especially useful if one group has only
  little observations, or if groups are not balanced.
  Higher numbers of \code{boot.n} will give a more precise estimate of the CI,
  but take more time to compute. 2000 is recommanded by Carpenter and
  Bithell.
}

\section{Warnings}{
  If \code{boot.stratified=FALSE} and the sample has a large imbalance between
  cases and controls, it could happen that one or more of the replicates
  contains no case or control observation, or that there are not enough
  points for smoothing, producing a \code{NA} area.
  The warning \dQuote{NA value(s) produced during bootstrap were ignored.}
  will be issued and the observation will be ignored. If you have a large
  imbalance in your sample, it could be safer to keep
  \code{boot.stratified=TRUE}.
}

\value{
  A list of length 2 and class \dQuote{ci.thresholds}, with the confidence
  intervals of the CI and the following items:
  \item{specificity}{a matrix of CI for the specificity. Row (names) are the
	thresholds, the first column the lower bound, the 2nd column the
	median and the 3rd column the upper bound.
  }
  \item{sensitivity}{same than specificity.}
  Additionally, the list has the following attributes: 
  \item{conf.level}{the width of the CI, in fraction.}
  \item{boot.n}{the number of bootstrap replicates.}
  \item{boot.stratified}{whether or not the bootstrapping was stratified.}
  \item{thresholds}{the thresholds, as given in argument.}
  \item{roc}{the object of class \dQuote{\link{roc}} that was used to
	compute the CI.
  }
}

\references{
  Tom Fawcett (2006) ``An introduction to ROC analysis''. \emph{Pattern
    Recognition Letters} \bold{27}, 861--874. DOI:
  10.1016/j.patrec.2005.10.010.

  James Carpenter and John Bithell (2000) ``Bootstrap condence intervals:
  when, which, what? A practical guide for medical statisticians''.
  \emph{Statistics in Medicine} \bold{19}, 1141--1164.
}

\seealso{
  \code{\link{roc}},
  \code{\link{ci}}
}

\examples{
data(aSAH)

# Syntax (response, predictor):
ci.thresholds(aSAH$outcome, aSAH$s100b)

# With a roc object:
rocobj <- roc(aSAH$outcome, aSAH$s100b)
ci.thresholds(rocobj)

# Customized bootstrap and specific thresholds:
ci.thresholds(aSAH$outcome, aSAH$s100b,
              boot.n=500, conf.level=0.9, stratified=FALSE,
              thresholds=c(0.5, 1, 2))

# Alternatively, you can get the CI directly from roc():
rocobj <- roc(aSAH$outcome,
              aSAH$s100b, ci=TRUE, of="thresholds")
rocobj$ci

# Plotting the CI
plot(rocobj)
plot(rocobj$ci)
}

\keyword{univar}
\keyword{nonparametric}
\keyword{utilities}
\keyword{roc}
