# pROC: Tools Receiver operating characteristic (ROC curves) with
# (partial) area under the curve, confidence intervals and comparison. 
# Copyright (C) 2010, 2011 Xavier Robin, Alexandre Hainard, Natacha Turck,
# Natalia Tiberti, Frédérique Lisacek, Jean-Charles Sanchez
# and Markus Müller
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

smooth <- function(x, ...)
  UseMethod("smooth", x, ...)

smooth.default <- function(x, ...) {
  getFunction("smooth", where="splus")(x, ...)
}

smooth.smooth.roc <- function(smooth.roc, ...) {
  roc <- attr(smooth.roc, "roc")
  if (is.null(roc))
    stop("Cannot smooth a ROC curve generated directly with numeric 'density.controls' and 'density.cases'.")
  smooth.roc(roc, ...)
}

smooth.roc <- function(roc, method = c("binormal", "density", "fitdistr"), n = 512, bandwidth = "nrd",
                       # method=density
                       density = NULL, density.controls = NULL, density.cases = NULL,
                       reuse.auc = TRUE, reuse.ci = FALSE,
                       ...) {
  if (is.ordered(roc$original.predictor))
    stop("Only ROC curves of numeric predictors can be smoothed.")

  method <- match.arg(method)

  if (mode(method) == "function") {
    sesp <- method(roc=roc, n=n, bandwidth=bandwidth, density=density, density.controls=density.controls, density.cases=density.cases, ...)
    if (mode(sesp) != "list" || ! identical(sort(names(sesp)), c("sensitivities", "specificities")))
      stop("'method' function did not return a list with two elements named 'sensitivities' and 'specificities'.")
    if (length(sesp$sensitivities) != length(sesp$specificities))
      stop("'method' function did return a list with two elements of different length.")
    if (! is.numeric(sesp$specificities) || ! is.numeric(sesp$sensitivities))
      stop("'method' function did not return a list of numeric vectors.")
    if (any(c(sesp$specificities, sesp$sensitivities) < 0) || any(c(sesp$specificities, sesp$sensitivities) > ifelse (roc$percent, 100, 1)))
      stop("'method' function returned vector with invalid values (< 0 or > 1 or 100).")
  }
  else {
    if (method == "binormal")
      sesp <- smooth.roc.binormal(roc, n)
    if (method == "fitdistr")
      sesp <- smooth.roc.fitdistr(roc, n, density.controls, density.cases, ...)
    if (method == "density") {
      sesp <- smooth.roc.density(roc, n, density.controls, density.cases, bandwidth, ...)
    }
  }

  class(sesp) <- "smooth.roc"
  sesp <- sort.roc(sesp) # sort se and sp
  # anchor SE/SP at 0/100
  sesp$specificities <- c(0, as.vector(sesp$specificities), ifelse(roc$percent, 100, 1))
  sesp$sensitivities <- c(ifelse(roc$percent, 100, 1), as.vector(sesp$sensitivities), 0)
  attr(sesp, "roc") <- roc # keep the original roc. May be useful in CI.
  sesp$percent <- roc$percent # keep some basic roc specifications
  sesp$direction <- roc$direction
  sesp$call <- match.call()
  # keep smoothing arguments (for print and bootstrap)
  sesp$smoothing.args <- list(...)
  sesp$smoothing.args$method <- method
  sesp$smoothing.args$n <- n
  sesp$smoothing.args$density.controls <- density.controls
  sesp$smoothing.args$density.cases <- density.cases
  sesp$smoothing.args$bandwidth <- bandwidth
  # complete fit.controls/cases if a function was passed as densfun
  if (method == "fitdistr") {
    if (is.null(sesp$fit.controls$densfun)) {
      if (missing(density.controls))
        sesp$fit.controls$densfun <- deparse(substitute(density))
      else
        sesp$fit.controls$densfun <- deparse(substitute(density.controls))
    }
    if (is.null(sesp$fit.cases$densfun)) {
      if (missing(density.cases))
        sesp$fit.cases$densfun <- deparse(substitute(density))
      else
        sesp$fit.cases$densfun <- deparse(substitute(density.cases))
    }
  }

  # if there was an auc and a ci, re-do them
  if (!is.null(roc$auc) && reuse.auc) {
    args <- attributes(roc$auc)
    args$roc <- NULL
    args$smooth.roc <- sesp
    sesp$auc <- do.call("auc.smooth.roc", args)
  }
  if (!is.null(roc$ci) && reuse.ci){
    args <- attributes(roc$ci)
    args$roc <- NULL
    args$smooth.roc <- sesp
    sesp$ci <- do.call(paste(class(roc$ci), "smooth.roc", sep="."), args)
  }

  return(sesp)
}

smooth.roc.density <- function(roc, n, density.controls, density.cases, bandwidth,
                               # catch args for density
                               cut = 3, adjust = 1, window = kernel, kernel = "gaussian",
                               percent = roc$percent, direction = roc$direction,
                               ...) {
  if (!is.numeric(density.controls) || !is.numeric(density.cases)) {
    if (is.character(bandwidth))
      bandwidth <- getFunction(paste("bandwidth", bandwidth, sep="."))(roc$predictor)
    bandwidth <- bandwidth * adjust
    from <- min(roc$predictor) - (cut * bandwidth)
    to <- max(roc$predictor) + (cut * bandwidth)
  }
  if (is.null(density.controls)) {
    if (mode(density.controls) == "function") {
      density.controls <- density.controls(roc$controls, n=n, from=from, to=to, width=bandwidth, window=window, ...)
      if (! is.numeric(density.controls)) {
        if (is.list(density.controls) && !is.null(density.controls$y) && is.numeric(density.controls$y))
          density.controls <- density.controls$y
        else
          stop("The 'density' function must return a numeric vector or a list with a 'y' item.")
      }
    }
    else if (is.null(density.controls))
      density.controls <- density(roc$controls, n=n, from=from, to=to, width=bandwidth, window=window)$y
    else if (! is.numeric(density.controls))
      stop("'density.controls' must be either NULL, a function or numeric values of density (over the y axis).")
  }
  if (is.null(density.cases)) {
    if (mode(density.cases) == "function") {
      density.cases <- density.cases(roc$cases, n=n, from=from, to=to, width=bandwidth, window=window, ...)
      if (! is.numeric(density.cases)) {
        if (is.list(density.cases) && !is.null(density.cases$y) && is.numeric(density.cases$y))
          density.cases <- density.cases$y
        else
          stop("The 'density' function must return a numeric vector or a list with a 'y' item.")
      }
    }
    else if (is.null(density.cases))
      density.cases <- density(roc$cases, n=n, from=from, to=to, width=bandwidth, window=window)$y
    else if (! is.numeric(density.cases))
      stop("'density.cases' must be either NULL, a function or numeric values of density (over the y axis).")
  }
  if (length(density.controls) != length(density.cases))
    stop("Length of 'density.controls' and 'density.cases' differ.")

  perfs <- sapply((1:length(density.controls))+.5, roc.utils.perfs.dens, x=(1:length(density.controls))+.5, dens.controls=density.controls, dens.cases=density.cases, direction=direction)

  return(list(sensitivities = perfs[2,] * ifelse(percent, 100, 1),
              specificities = perfs[1,] * ifelse(percent, 100, 1)))
}

smooth.roc.binormal <- function(roc, n) {
  df <- data.frame(sp=qnorm(roc$sp * ifelse(roc$percent, 1/100, 1)), se=qnorm(roc$se * ifelse(roc$percent, 1/100, 1)))
  df <- df[apply(df, 1, function(x) all(is.finite(x))),]
  if (dim(df)[1] <= 1) # ROC curve or with only 1 point
    stop("ROC curve not smoothable (not enough points).")
  model <- try(lm(sp~se, df))
  if (is(model, "Error"))
    stop("Error with 'lm' in smoothing. The ROC curve has probably not enough points to be smoothed with the 'binormal' model. Try 'density' or 'fitdistr' instead. Original error:\n", model)
  se <- qnorm(seq(0, 1, 1/(n-1)))
  sp <- predict(model, data.frame(se))

  return(list(sensitivities = pnorm(se) * ifelse(roc$percent, 100, 1),
              specificities = pnorm(sp) * ifelse(roc$percent, 100, 1)))
}

smooth.roc.fitdistr <- function(roc, n, densfun.controls, densfun.cases, ...) {
  densfuns.list <- list(exponential = "dexp", "lognormal" = "dlnorm", "log-normal" = "dlnorm", normal = "dnorm", uniform = "dunif")

  if (is.null(densfun.controls))
    densfun.controls <- "normal"
  else if (is.character(densfun.controls))
    densfun.controls <- match.arg(densfun.controls, names(densfuns.list))

  if (is.null(densfun.cases))
    densfun.cases <- "normal"
  else if (is.character(densfun.cases))
    densfun.cases <- match.arg(densfun.cases, names(densfuns.list))

  # get the actual function in densfun.*.fun
  if (is.character(densfun.controls))
    densfun.controls.fun <- getFunction(densfuns.list[[densfun.controls]])
  else
    densfun.controls.fun <- densfun.controls
  if (is.character(densfun.cases))
    densfun.cases.fun <- getFunction(densfuns.list[[densfun.cases]])
  else
    densfun.cases.fun <- densfun.cases

  # fit distributions
  fit.controls <- do.call("fitdistr", list(x=roc$controls, densfun=densfun.controls))
  fit.cases <- do.call("fitdistr", list(x=roc$cases, densfun=densfun.cases))

  # store function name in fitting results
  if (mode(densfun.controls) != "function")
    fit.controls$densfun <- densfun.controls
  if (mode(densfun.cases) != "function")
    fit.cases$densfun <- densfun.cases

  # generate smoothed densities
  x <- seq(min(c(roc$controls, roc$cases)), max(c(roc$controls, roc$cases)), length.out=512)
  density.controls <- do.call("densfun.controls.fun", c(list(x=x), fit.controls$estimate))
  density.cases <- do.call("densfun.cases.fun", c(list(x=x), fit.cases$estimate))

  perfs <- sapply(roc.utils.thresholds(x), roc.utils.perfs.dens, x=x, dens.controls=density.controls, dens.cases=density.cases, direction=roc$direction)

  return(list(sensitivities = perfs[2,] * ifelse(roc$percent, 100, 1),
              specificities = perfs[1,] * ifelse(roc$percent, 100, 1),
              fit.controls=fit.controls, fit.cases=fit.cases))
}

# emulate not working MASS fitdistr function
fitdistr <- function (x, densfun=c("normal", "lognormal", "log-normal", "exponential", "uniform")) {
  densfun <- match.arg(densfun)
  if (densfun == "normal")
    return(list(estimate = c(mean=mean(x), sd=sd(x))))
  else if (densfun == "log-normal" || densfun == "lognormal")
    return(list(estimate = c(meanlog=mean(log(x)), sdlog=sd(log(x)))))
  else if (densfun == "exponential")
    return(list(estimate = c(rate=1/mean(x))))
  else if (densfun == "uniform")
    return(list(estimate = c(min = min(x), max = max(x))))
}
