# pROC: Tools Receiver operating characteristic (ROC curves) with
# (partial) area under the curve, confidence intervals and comparison. 
# Copyright (C) 2010, 2011 Xavier Robin, Alexandre Hainard, Natacha Turck,
# Natalia Tiberti, Frédérique Lisacek, Jean-Charles Sanchez
# and Markus Müller
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

venkatraman.paired.test <- function(roc1, roc2, boot.n) {
  X <- roc1$predictor
  Y <- roc2$predictor
  R <- rank(X)
  S <- rank(Y)
  D <- roc1$response # because roc1&roc2 are paired

  E <- venkatraman.paired.stat(R, S, D, roc1$levels)
  EP <- sapply(1:boot.n, venkatraman.paired.permutation, R=R, S=S, D=D, levels=roc1$levels)
  return(list(E, EP))
}

venkatraman.unpaired.test <- function(roc1, roc2, boot.n) {
  X <- roc1$predictor
  Y <- roc2$predictor
  R <- rank(X)
  S <- rank(Y)
  D1<- roc1$response
  D2 <- roc2$response
  mp <- (sum(D1 == roc1$levels[2]) + sum(D2 == roc2$levels[2]))/(length(D1) + length(D1)) # mixing proportion, kappa

  E <- venkatraman.unpaired.stat(R, S, D1, D2, roc1$levels, roc2$levels, mp)
  EP <- sapply(1:boot.n, venkatraman.unpaired.permutation, R=R, S=S, D1=D1, D2=D2, levels1=roc1$levels, levels2=roc2$levels, mp=mp)
  return(list(E, EP))
}

venkatraman.paired.permutation <- function(x, R, S, D, levels) { # x: ignored, to be compatible with sapply
  # Break ties
  R2 <- R + runif(length(D)) - 0.5 # Add small amount of random but keep same mean
  S2 <- S + runif(length(D)) - 0.5

  # Permutation
  q <- 1 - round(runif(length(D)))
  R3 <- R2 * q + (1 - q) * S
  S3 <- S2 * q + (1 - q) * R

  return(venkatraman.paired.stat(rank(R3), rank(S3), D, levels))
}

venkatraman.unpaired.permutation <- function(x, R, S, D1, D2, levels1, levels2, mp) {
  # Break ties
  R <- R + runif(length(D1)) - 0.5 # Add small amount of random but keep same mean
  S <- S + runif(length(D2)) - 0.5

  R.controls <- R[D1==levels1[1]]
  R.cases <- R[D1==levels1[2]]
  S.controls <- S[D2==levels2[1]]
  S.cases <- S[D2==levels2[2]]

  # Permutation
  controls <- sample(c(R.controls, S.controls))
  cases <- sample(c(R.cases, S.cases))
  R[D1==levels1[1]] <- controls[1:length(R.controls)]
  S[D2==levels2[1]] <- controls[(length(R.controls)+1):length(controls)]
  R[D1==levels1[2]] <- cases[1:length(R.cases)]
  S[D2==levels2[2]] <- cases[(length(R.cases)+1):length(cases)]

  return(venkatraman.unpaired.stat(rank(R), rank(S), D1, D2, levels1, levels2, mp))
}

venkatraman.paired.stat <- function(R, S, D, levels) {
  R.controls <- R[D==levels[1]]
  R.cases <- R[D==levels[2]]
  S.controls <- S[D==levels[1]]
  S.cases <- S[D==levels[2]]
  n <- length(D)

  R.fn <- sapply(1:n, function(x, R.cases) sum(R.cases <= x), R.cases=R.cases)
  R.fp <- sapply(1:n, function(x, R.controls) sum(R.controls > x), R.controls=R.controls)
  S.fn <- sapply(1:n, function(x, S.cases) sum(S.cases <= x), S.cases=S.cases)
  S.fp <- sapply(1:n, function(x, S.controls) sum(S.controls > x), S.controls=S.controls)

  return(sum(abs((S.fn + S.fp) - (R.fn + R.fp))))
}

venkatraman.unpaired.stat <- function(R, S, D1, D2, levels1, levels2, mp) {
  R.controls <- R[D1==levels1[1]]
  R.cases <- R[D1==levels1[2]]
  S.controls <- S[D2==levels2[1]]
  S.cases <- S[D2==levels2[2]]
  n <- length(D1)
  m <- length(D2)

  R.fx <- sapply(1:n, function(x, R.cases) sum(R.cases <= x), R.cases=R.cases) / length(R.cases)
  R.gx <- sapply(1:n, function(x, R.controls) sum(R.controls <= x), R.controls=R.controls) / length(R.controls)
  S.fx <- sapply(1:m, function(x, S.cases) sum(S.cases <= x), S.cases=S.cases) / length(S.cases)
  S.gx <- sapply(1:m, function(x, S.controls) sum(S.controls <= x), S.controls=S.controls) / length(S.controls)
  R.p <- mp*R.fx + (1 - mp)*R.gx
  S.p <- mp*S.fx + (1 - mp)*S.gx
  R.exp <- mp*R.fx + (1 - mp)*(1-R.gx)
  S.exp <- mp*S.fx + (1 - mp)*(1-S.gx)

  # Do the integration
  x <- sort(c(R.p, S.p))
  R.f <- approx(R.p, R.exp, xout=x)$y
  S.f <- approx(S.p, S.exp, xout=x)$y
  y <- abs(R.f - S.f)
  #trapezoid integration:
  idx <- 2:length(x)
  integral <- sum(((y[idx] + y[idx-1]) * (x[idx] - x[idx-1])) / 2, na.rm=TRUE) # remove NA that can appear in the borders
  return(integral)
}
