# pROC: Tools Receiver operating characteristic (ROC curves) with
# (partial) area under the curve, confidence intervals and comparison. 
# Copyright (C) 2010, 2011 Xavier Robin, Alexandre Hainard, Natacha Turck,
# Natalia Tiberti, Frédérique Lisacek, Jean-Charles Sanchez
# and Markus Müller
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

plot.roc <- function(x, ...) {
  UseMethod("plot.roc")
}

plot.roc.formula <- function(x, data, ...) {
  roc <- roc(x, data, plot=TRUE, ...)
  roc$call <- match.call()
  return(roc)
}

plot.roc.default <- function(x, predictor, ...) {
  roc <- roc(x, predictor, plot=TRUE, ...)
  roc$call <- match.call()
  return(roc)
}

plot.smooth.roc <- function(x, ...) {
  plot.roc.roc(x, ...) # force usage of plot.roc.roc: only print.thres not working
}
plot.roc.smooth.roc <- plot.smooth.roc

plot.roc.roc <- function(x,
                         add=FALSE,
                         reuse.auc=TRUE,
                         xlim=if(x$percent){c(0, 100)} else{c(0, 1)},
                         ylim=if(x$percent){c(0, 100)} else{c(0, 1)},
                         xlab=ifelse(x$percent, "1 - Specificity (%)", "1 - Specificity"),
                         ylab=ifelse(x$percent, "Sensitivity (%)", "Sensitivity"),
                         asp=1,
                         width=if(is.ui.app("s+gui")) 8.5 else if(is.ui.app("s+java")) 600,
                         height=if(is.ui.app("s+gui")) 8.5 else if(is.ui.app("s+java")) 600,
                         # col, lty and lwd for the ROC line only
                         col=par("col"),
                         lty=par("lty"),
                         lwd=3,
                         type="l",
                         # Identity line
                         identity=!add,
                         identity.col="darkgrey",
                         identity.lty=1,
                         identity.lwd=1,
                         # Print the thresholds on the plot
                         print.thres=FALSE,
                         print.thres.pch=16,
                         print.thres.col="black",
                         print.thres.pattern=ifelse(x$percent, "%.1f (%.1f%%, %.1f%%)", "%.3f (%.3f, %.3f)"),
                         print.thres.cex=par("cex"),
                         print.thres.pattern.cex=print.thres.cex,
                         # Print the AUC on the plot
                         print.auc=FALSE,
                         print.auc.pattern=NULL,
                         print.auc.x=ifelse(x$percent, 50, .5), 
                         print.auc.y=ifelse(x$percent, 50, .5),
                         print.auc.col=col,
                         print.auc.cex=par("cex"),
                         # Grid
                         grid=FALSE,
                         grid.v={
                           if(is.logical(grid) && grid[1]==TRUE){seq(0, 1, 0.1) * ifelse(x$percent, 100, 1)}
                           else if(is.numeric(grid)) {seq(0, ifelse(x$percent, 100, 1), grid[1])}
                           else {NULL}
                         },
                         grid.h={
                           if (length(grid) == 1) {grid.v}
                           else if (is.logical(grid) && grid[2]==TRUE){seq(0, 1, 0.1) * ifelse(x$percent, 100, 1)}
                           else if(is.numeric(grid)) {seq(0, ifelse(x$percent, 100, 1), grid[2])}
                           else {NULL}
                         },
                         # for grid.lty, grid.lwd and grid.col, a length 2 value specifies both values for vertical (1) and horizontal (2) grid
                         grid.lty=2,
                         grid.lwd=1,
                         grid.col="#DDDDDD",
                         # Polygon for the auc
                         auc.polygon=FALSE,
                         auc.polygon.col="gainsboro", # Other arguments can be passed to polygon() using "..." (for these two we cannot)
                         auc.polygon.border="#000000",
                         auc.polygon.density=-1,
                         auc.polygon.angle=45,
                         # Should we show the maximum possible area as another polygon?
                         max.auc.polygon=FALSE,
                         max.auc.polygon.col="#EEEEEE", # Other arguments can be passed to polygon() using "..." (for these two we cannot)
                         max.auc.polygon.border="#000000",
                         max.auc.polygon.density=-1,
                         max.auc.polygon.angle=45,
                         # Confidence interval
                         ci=!is.null(x$ci),
                         ci.type=c("bars", "shape", "no"),
                         ci.col=ifelse(ci.type=="bars", par("fg"), "gainsboro"),
                         ...
                         ) {
  percent <- x$percent
  
  if (max.auc.polygon | auc.polygon | print.auc) {# we need the auc here
    if (is.null(x$auc) | !reuse.auc)
      x$auc <- auc(x, ...)
    partial.auc <- attr(x$auc, "partial.auc")
    partial.auc.focus <- attr(x$auc, "partial.auc.focus")
  }

  # compute a reasonable default for print.auc.pattern if required
  if (print.auc & is.null(print.auc.pattern)) {
    print.auc.pattern <- ifelse(identical(partial.auc, FALSE), "AUC: ", "Partial AUC: ")
    print.auc.pattern <- paste(print.auc.pattern, ifelse(percent, "%.1f%%", "%.3f"), sep="")
    if (ci && class(x$ci) == "ci.auc")
      print.auc.pattern <- paste(print.auc.pattern, " (", ifelse(percent, "%.1f%%", "%.3f"), "-", ifelse(percent, "%.1f%%", "%.3f"), ")",sep="")
  }
    
  # get and sort the sensitivities and specificities
  se <- rev(sort(x$se))
  sp <- sort(x$sp)
  if (!add) {
    if (dev.cur()==1 && is.ui.app("s+gui")) { # prefer a square plot. If no device exists, start a graphsheet with width/height defaulting to square
      graphsheet(width, height, params=list(...))
      par(mar=c(4, 4, 2, 2)+.1)
      par(mgp=c(2.5, 1, 0))
    }
    else if (dev.cur()==1 && is.ui.app("s+java")) { # similar but for java
      java.graph(width=width, height=height, ...)
      par(mar=c(4, 4, 2, 2)+.1)
      par(mgp=c(2.5, 1, 0))
    }
  # type="n" to plot background lines and polygon shapes first. We will add the line later
    plot(ifelse(percent, 100, 1) - x$sp, x$se, xlab=xlab, ylab=ylab, type="n", xlim=xlim, ylim=ylim, lwd=lwd, asp=asp, ...)
  }

  # Plot the grid
  # make sure grid.lty, grid.lwd and grid.col are at least of length 2
  grid.lty <- rep(grid.lty, length.out=2)
  grid.lwd <- rep(grid.lwd, length.out=2)
  grid.col <- rep(grid.col, length.out=2)
  if (!is.null(grid.v)) {
    abline(v=ifelse(percent, 100, 1) - grid.v, lty=grid.lty[1], col=grid.col[1], lwd=grid.lwd[1], ...)
  }
  if (!is.null(grid.h)) {
    abline(h=grid.h, lty=grid.lty[2], col=grid.col[2], lwd=grid.lwd[2], ...)
  }

  # Plot the polygon displaying the maximal area
  if (max.auc.polygon) {
    if (identical(partial.auc, FALSE)) {
      map.y <- c(0, 1, 1, 0) * ifelse(percent, 100, 1)
      map.x <- c(1, 1, 0, 0) * ifelse(percent, 100, 1)
    }
    else {
      if (partial.auc.focus=="sensitivity") {
        map.y <- c(partial.auc[2], partial.auc[2], partial.auc[1], partial.auc[1]) 
        map.x <- c(0, 1, 1, 0) * ifelse(percent, 100, 1) 
      }
      else {
        map.y <- c(0, 1, 1, 0) * ifelse(percent, 100, 1) 
        map.x <- c(partial.auc[2], partial.auc[2], partial.auc[1], partial.auc[1])
      }
    }
    polygon(ifelse(percent, 100, 1) - map.x, map.y, col=max.auc.polygon.col, border=max.auc.polygon.border, density=max.auc.polygon.density, angle=max.auc.polygon.angle)
  }
  # Plot the ci shape
  if (ci && class(x$ci) != "ci.auc") {
    ci.type <- match.arg(ci.type)
    if (ci.type=="shape")
      plot(x$ci, type="shape", col=ci.col, no.roc=TRUE, ...)
  }
  # Plot the polygon displaying the actual area
  if (auc.polygon) {
    if (identical(partial.auc, FALSE)) {
      polygon(ifelse(percent, 100, 1) - c(sp, 0), c(se, 0), col=auc.polygon.col, border=auc.polygon.border, density=auc.polygon.density, angle=auc.polygon.angle)
    }
    else {
      if (partial.auc.focus == "sensitivity") {
        x.all <- rev(se)
        y.all <- rev(sp)
      }
      else {
        x.all <- sp
        y.all <- se
      }
      # find the SEs and SPs in the interval
      x.int <- x.all[x.all <= partial.auc[1] & x.all >= partial.auc[2]]
      y.int <- y.all[x.all <= partial.auc[1] & x.all >= partial.auc[2]]
        # if the upper limit is not exactly present in SPs, interpolate
      if (!(partial.auc[1] %in% x.int)) {
        x.int <- c(x.int, partial.auc[1])
        # find the limit indices
        idx.out <- match(FALSE, x.all < partial.auc[1])
        idx.in <- idx.out - 1
        # interpolate y
        proportion.start <- (partial.auc[1] - x.all[idx.out]) / (x.all[idx.in] - x.all[idx.out])
        y.start <- y.all[idx.out] - proportion.start * (y.all[idx.out] - y.all[idx.in])
        y.int <- c(y.int, y.start)
      }
        # if the lower limit is not exactly present in SPs, interpolate
      if (!(partial.auc[2] %in% x.int)) {
        x.int <- c(partial.auc[2], x.int)
        # find the limit indices
        idx.out <- length(x.all) - match(TRUE, rev(x.all) < partial.auc[2]) + 1
        idx.in <- idx.out + 1
        # interpolate y
        proportion.end <- (x.all[idx.in] - partial.auc[2]) / (x.all[idx.in] - x.all[idx.out])
        y.end <- y.all[idx.in] + proportion.end * (y.all[idx.out] - y.all[idx.in])
        y.int <- c(y.end, y.int)
      }
      # anchor to baseline
      x.int <- c(partial.auc[2], x.int, partial.auc[1])
      y.int <- c(0, y.int, 0)
      if (partial.auc.focus == "sensitivity") {
        # for SE, invert x and y again
        polygon(ifelse(percent, 100, 1) - y.int, x.int, col=auc.polygon.col, border=auc.polygon.border, density=auc.polygon.density, angle=auc.polygon.angle)
      }
      else {
        polygon(ifelse(percent, 100, 1) - x.int, y.int, col=auc.polygon.col, border=auc.polygon.border, density=auc.polygon.density, angle=auc.polygon.angle)
      }
    }
  }
  # Identity line
  if (identity) abline(0, 1, col=identity.col, lwd=identity.lwd, lty=identity.lty, ...)
  # Actually plot the ROC curve
  lines(ifelse(percent, 100, 1) - sp, se, type=type, lwd=lwd, col=col, lty=lty, ...)
  # Plot the ci bars
  if (ci && class(x$ci) != "ci.auc") {
    if (ci.type=="bars")
      plot(x$ci, type="bars", col=ci.col, ...)
  }
  # Print the thresholds on the curve if print.thres is TRUE
  if (isTRUE(print.thres))
    print.thres <- "best"
  if (is.character(print.thres))
    print.thres <- match.arg(print.thres, c("no", "all", "local maximas", "best"))
  if (class(x) == "smooth.roc") {
    if (is.numeric(print.thres))
      stop("Numeric 'print.thres' unsupported on a smoothed ROC plot.")
    else if (print.thres == "all" || print.thres == "local maximas")
      stop("'all' and 'local maximas' 'print.thres' unsupported on a smoothed ROC plot.") 
    else if (print.thres == "best") {
      co <- coords(x, print.thres)
      if (class(co) == "matrix") {
        points(ifelse(percent, 100, 1) - co[1,], co[2,], pch=print.thres.pch, cex=print.thres.cex, col=print.thres.col, ...)
        text(ifelse(percent, 100, 1) - co[1,] + ifelse(percent, 2, .02), co[2,] - ifelse(percent, 2, .02), sprintf(print.thres.pattern, NA, co[1,], co[2,]), cex=print.thres.pattern.cex, col=print.thres.col, ...)
      }
      else {
        points(ifelse(percent, 100, 1) - co[1], co[2], pch=print.thres.pch, cex=print.thres.cex, col=print.thres.col, ...)
        text(ifelse(percent, 100, 1) - co[1] + ifelse(percent, 2, .02), co[2] - ifelse(percent, 2, .02), sprintf(print.thres.pattern, NA, co[1], co[2]), cex=print.thres.pattern.cex, adj=0, col=print.thres.col, ...)
      }
    } # else print.thres == no > do nothing
  }
  else if (is.numeric(print.thres) || is.character(print.thres)) {
    if (is.character(print.thres) && print.thres == "no") {} # do nothing
    else {
      co <- coords(x, print.thres)
      if (class(co) == "matrix") {
        points(ifelse(percent, 100, 1) - co[2,], co[3,], pch=print.thres.pch, cex=print.thres.cex, col=print.thres.col, ...)
        text(ifelse(percent, 100, 1) - co[2,] + ifelse(percent, 2, .02), co[3,] - ifelse(percent, 2, .02), sprintf(print.thres.pattern, co[1,], co[2,], co[3,]), cex=print.thres.pattern.cex, col=print.thres.col, ...)
      }
      else {
        points(ifelse(percent, 100, 1) - co[2], co[3], pch=print.thres.pch, cex=print.thres.cex, col=print.thres.col, ...)
        text(ifelse(percent, 100, 1) - co[2] + ifelse(percent, 2, .02), co[3] - ifelse(percent, 2, .02), sprintf(print.thres.pattern, co[1], co[2], co[3]), cex=print.thres.pattern.cex, adj=0, col=print.thres.col, ...)
      }
    }
  }

  # Print the AUC on the plot
  if (print.auc) {
    if (ci && class(x$ci) == "ci.auc") {
      labels <- sprintf(print.auc.pattern, x$auc, x$ci[1], x$ci[3])
      text(print.auc.x, print.auc.y, labels, adj=0, cex=print.auc.cex, col=print.auc.col, ...)
    }
    else
      labels <- sprintf(print.auc.pattern, x$auc)
    text(print.auc.x, print.auc.y, labels, adj=0, cex=print.auc.cex, col=print.auc.col, ...)
  }
  
  return(x)
}
